/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.backup;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.NoSuchElementException;
import org.neo4j.backup.BackupExtensionService;
import org.neo4j.backup.OnlineBackup;
import org.neo4j.com.ComException;
import org.neo4j.graphdb.TransactionFailureException;
import org.neo4j.helpers.Args;
import org.neo4j.helpers.Service;
import org.neo4j.kernel.impl.storemigration.LogFiles;
import org.neo4j.kernel.impl.storemigration.StoreFiles;
import org.neo4j.kernel.impl.storemigration.UpgradeNotAllowedByConfigurationException;

public class BackupTool {
    private static final String TO = "to";
    private static final String FROM = "from";
    private static final String INCREMENTAL = "incremental";
    private static final String FULL = "full";
    private static final String VERIFY = "verify";
    public static final String DEFAULT_SCHEME = "single";

    public static void main(String[] args) {
        Args arguments = new Args(args);
        BackupTool.checkArguments(arguments);
        boolean full = arguments.has(FULL);
        String from = arguments.get(FROM, null);
        String to = arguments.get(TO, null);
        boolean verify = arguments.getBoolean(VERIFY, Boolean.valueOf(true), Boolean.valueOf(true));
        URI backupURI = null;
        try {
            backupURI = new URI(from);
        }
        catch (URISyntaxException e) {
            BackupTool.exitAbnormally("Please properly specify a location to backup as a valid URI in the form <scheme>://<host>[:port], where scheme is the target database's running mode, eg ha");
        }
        String module = backupURI.getScheme();
        BackupExtensionService service = null;
        if (module != null && !DEFAULT_SCHEME.equals(module)) {
            try {
                service = (BackupExtensionService)Service.load(BackupExtensionService.class, (String)module);
            }
            catch (NoSuchElementException e) {
                BackupTool.exitAbnormally(String.format("%s was specified as a backup module but it was not found. Please make sure that the implementing service is on the classpath.", module));
            }
        }
        if (service != null) {
            backupURI = service.resolve(backupURI, arguments);
        }
        BackupTool.doBackup(full, backupURI, to, verify);
    }

    private static void checkArguments(Args arguments) {
        boolean incremental;
        boolean full = arguments.has(FULL);
        if (full & (incremental = arguments.has(INCREMENTAL)) || !(full | incremental)) {
            BackupTool.exitAbnormally("Specify either " + BackupTool.dash(FULL) + " or " + BackupTool.dash(INCREMENTAL));
        }
        if (arguments.get(FROM, null) == null) {
            BackupTool.exitAbnormally("Please specify " + BackupTool.dash(FROM) + ", examples:\n" + "  " + BackupTool.dash(FROM) + " single://192.168.1.34\n" + "  " + BackupTool.dash(FROM) + " single://192.168.1.34:1234\n" + "  " + BackupTool.dash(FROM) + " ha://192.168.1.15:2181\n" + "  " + BackupTool.dash(FROM) + " ha://192.168.1.15:2181,192.168.1.16:2181");
        }
        if (arguments.get(TO, null) == null) {
            BackupTool.exitAbnormally("Specify target location with " + BackupTool.dash(TO) + " <target-directory>");
        }
    }

    private static void doBackup(boolean trueForFullFalseForIncremental, URI from, String to, boolean verify) {
        if (trueForFullFalseForIncremental) {
            BackupTool.doBackupFull(from, to, verify);
        } else {
            BackupTool.doBackupIncremental(from, to, verify);
        }
        System.out.println("Done");
    }

    private static void doBackupFull(URI from, String to, boolean verify) {
        System.out.println("Performing full backup from '" + from + "'");
        OnlineBackup backup = BackupTool.newOnlineBackup(from);
        try {
            backup.full(to, verify);
        }
        catch (ComException e) {
            BackupTool.exitAbnormally("Couldn't connect to '" + from + "'", (Exception)((Object)e));
        }
    }

    private static void doBackupIncremental(URI from, String to, boolean verify) {
        System.out.println("Performing incremental backup from '" + from + "'");
        OnlineBackup backup = BackupTool.newOnlineBackup(from);
        boolean failedBecauseOfStoreVersionMismatch = false;
        try {
            backup.incremental(to, verify);
        }
        catch (TransactionFailureException e) {
            if (e.getCause() instanceof UpgradeNotAllowedByConfigurationException) {
                failedBecauseOfStoreVersionMismatch = true;
            } else {
                BackupTool.exitAbnormally("TransactionFailureException from existing backup at '" + from + "'.", (Exception)((Object)e));
            }
        }
        catch (ComException e) {
            BackupTool.exitAbnormally("Couldn't connect to '" + from + "' ", (Exception)((Object)e));
        }
        if (failedBecauseOfStoreVersionMismatch) {
            System.out.println("The database present in the target directory is of an older version. Backing that up in target and performing a full backup from source");
            try {
                BackupTool.moveExistingDatabase(to);
            }
            catch (IOException e) {
                BackupTool.exitAbnormally("There was a problem moving the old database out of the way - cannot continue, aborting.", e);
            }
            BackupTool.doBackupFull(from, to, verify);
        }
    }

    private static void moveExistingDatabase(String to) throws IOException {
        File toDir = new File(to);
        File backupDir = new File(toDir, "old-version");
        if (!backupDir.mkdir()) {
            throw new IOException("Trouble making target backup directory " + backupDir.getAbsolutePath());
        }
        StoreFiles.move((File)toDir, (File)backupDir);
        LogFiles.move((File)toDir, (File)backupDir);
    }

    private static void exitAbnormally(String message, Exception ex) {
        System.out.println(message);
        ex.printStackTrace(System.out);
        System.exit(1);
    }

    private static void exitAbnormally(String message) {
        System.out.println(message);
        System.exit(1);
    }

    private static String dash(String name) {
        return "-" + name;
    }

    private static OnlineBackup newOnlineBackup(URI from) {
        String host = from.getHost();
        int port = from.getPort();
        if (port == -1) {
            return OnlineBackup.from(host);
        }
        return OnlineBackup.from(host, port);
    }
}

